<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       none
 * @since      1.0.0
 *
 * @package    Slms_Sso
 * @subpackage Slms_Sso/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Slms_Sso
 * @subpackage Slms_Sso/admin
 * @author     Art Dingal <artdingal@bitkea.com>
 */
class Slms_Sso_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The options name to be used in this plugin
	 *
	 * @since  	1.0.0
	 * @access 	private
	 * @var  	string 		$option_name 	Option name of this plugin
	 */
	private $option_name = 'slmssso';

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;
	private $fields;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Slms_Sso_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Slms_Sso_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/slms-sso-admin.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Slms_Sso_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Slms_Sso_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/slms-sso-admin.js', array( 'jquery' ), $this->version, false );

	}

	/**
	 * Add an options page under the Settings submenu
	 *
	 * @since  1.0.0
	 */
	public function add_options_page() {

		$this->plugin_screen_hook_suffix = add_menu_page(
			__( 'ScholarLMS Single Sign-On Settings', 'slms-sso' ),
			__( 'ScholarLMS', 'slms-sso' ),
			'manage_options',
			$this->plugin_name,
			array( $this, 'display_options_page' )
			,'dashicons-welcome-learn-more'
		);
		
		add_submenu_page(
        	null,
        	'UUID Sync',
        	'UUID Sync',
        	'manage_options',
        	'slms_uuid_sync',
        	array($this, 'slms_uuid_sync_page')
        );

	}
	/**
	 * Render the options page for plugin
	 *
	 * @since  1.0.0
	 */
	public function display_options_page() {
		include_once 'partials/slms-sso-admin-display.php';
	}
	/**
	 * Register all related settings of this plugin
	 *
	 * @since  1.0.0
	 */
	public function register_setting() {

		$ws = new Slms_Sso_Ws($this->plugin_name,$this->version);
		$this->fields = $ws->call('request_user_fields');

	    add_settings_section(
			$this->option_name . '_general',
			__( 'General', 'slms-sso' ),
			array( $this, $this->option_name . '_general_cb'),
			$this->plugin_name
		);

		add_settings_field(
	        $this->option_name . '_lms_url',
	        __( 'LMS URL', 'slms-sso' ),
	        array( $this, $this->option_name . '_lms_url_cb' ), 
	        $this->plugin_name,
	        $this->option_name . '_general',
	        array( 'label_for' => $this->option_name . '_lms_url' )
	    );

	    add_settings_field(
	        $this->option_name . '_ws_token', 
	        __('Webservice token', 'slms-sso'), 
	        array( $this, $this->option_name . '_ws_token_cb' ),
	        $this->plugin_name,
	        $this->option_name . '_general',
	        array( 'label_for' => $this->option_name . '_ws_token' )
	    );

	    add_settings_field(
	        $this->option_name . '_secret_key', 
	        __('Secret key', 'slms-sso'), 
	        array( $this, $this->option_name . '_secret_key_cb' ),
	        $this->plugin_name,
	        $this->option_name . '_general',
	        array( 'label_for' => $this->option_name . '_secret_key' )
	    );

	    add_settings_field(
	        $this->option_name . '_redirect_url', 
	        __('Redirect URL', 'slms-sso'), 
	        array( $this, $this->option_name . '_redirect_url_cb' ),
	        $this->plugin_name,
	        $this->option_name . '_general',
	        array( 'label_for' => $this->option_name . '_redirect_url' )
	    );

	    register_setting( 
	    	$this->plugin_name, 
	    	$this->option_name . '_lms_url'
	    );

	    register_setting( 
	    	$this->plugin_name, 
	    	$this->option_name . '_ws_token'
	    );

	    register_setting( 
	    	$this->plugin_name, 
	    	$this->option_name . '_secret_key'
	    );

	    register_setting( 
	    	$this->plugin_name, 
	    	$this->option_name . '_redirect_url'
	    );

	    add_settings_section(
			$this->option_name . '_field_mapping',
			__( 'User Profile Fields Mapping', 'slms-sso' ),
			array( $this, $this->option_name . '_field_mapping_cb' ),
			$this->plugin_name
		);

	    $this->register_custom_fields($this->fields);


	}

	/**
	 * Render the text for the general section
	 *
	 * @since  1.0.0
	 */
	public function slmssso_general_cb() {
		echo '<p>' . __( 'Refer our <a href="https://www.scholarlms.com/docs/wordpress-integration/" target="_blank">documentation page</a> on how to get these values from your ScholarLMS account.', 'slms-sso' ) . '</p>';

		if ($this->fields) {
			echo '<div class="notice notice-success"><p><strong>Success! The SSO is configured properly.</strong></p></div>';
		}else{
			echo '<div class="notice notice-error"><p><strong>Failed! The SSO was not configured properly. Please provide correct details and \'Save Changes\' again.</strong></p></div>';
		}

	}

	/**
	 * Render the text input field for lms url option
	 *
	 * @since  1.0.0
	 */
	public function slmssso_lms_url_cb(){
	    $slmssso_lms_url_value = get_option('slmssso_lms_url');
		printf('<input name="slmssso_lms_url" type="text" class="regular-text" value="%s"/>', $slmssso_lms_url_value);
	}

	/**
	 * Render the text input field for ws token option
	 *
	 * @since  1.0.0
	 */
	public function slmssso_ws_token_cb() {
	    $slmssso_ws_token_value = get_option('slmssso_ws_token');
		printf('<input name="slmssso_ws_token" type="text" class="regular-text" value="%s"/>', $slmssso_ws_token_value);
	}

	/**
	 * Render the text input field for secret key option
	 *
	 * @since  1.0.0
	 */
	public function slmssso_secret_key_cb() {
	    $slmssso_secret_key_value = get_option('slmssso_secret_key');
    	printf('<input name="slmssso_secret_key" type="text" class="regular-text" value="%s"/>', $slmssso_secret_key_value);
	}

	/**
	 * Render the text input field for redirect url option
	 *
	 * @since  1.0.0
	 */
	public function slmssso_redirect_url_cb() {
	    $slmssso_redirect_url_value = get_option('slmssso_redirect_url');
    	printf('<input name="slmssso_redirect_url" type="text" class="regular-text" value="%s"/>', $slmssso_redirect_url_value);
	}

	/**
	 * Render the text for the field mapping section
	 *
	 * @since  1.0.0
	 */
	public function slmssso_field_mapping_cb() {
		echo '<p>' . __( 'Refer our <a href="https://www.scholarlms.com/docs/wordpress-integration/" target="_blank">documentation page</a> on how to get these values from your ScholarLMS account.', 'slms-sso' ) . '</p>';

		if ($this->fields) {
			echo $this->get_user_meta_keys();
		}
	}

	public function slmssso_defaultfield_cb($args) {
		$value = htmlentities(get_option($args['name']));
		printf('<input name="%s" id="%s" type="text" class="regular-text" value="%s"/>', $args['name'],$args['id'],$value);
	}

	public function slmssso_customfield_cb($args) {
		$value = htmlentities(get_option($args['name']));
		printf('<input name="%s" id="%s" type="text" class="regular-text" value="%s"/>', $args['name'],$args['id'],$value);
	}

	public function register_custom_fields($fields){

		if (!$fields) {
			return 'failed to fetch fields';
		}

		foreach ($fields[0]['defaultfields'] as $key => $field) {
			$id =  $this->option_name . '_field_mapping_'.$field['name'];
			$title = $field['name'];
			add_settings_field(
		        $id, 
		        __($field['description'], 'slms-sso'), 
		        array( $this, $this->option_name .'_defaultfield_cb'),
		        $this->plugin_name,
		        $this->option_name . '_field_mapping',
		        array( 
		        	'label_for' => $id,
		        	'name' => $id,
		        	'id' => $id,
		        )
		    );

		    register_setting( 
		    	$this->plugin_name, 
		    	$id
		    );
		}

		foreach ($fields[0]['customfields'] as $key => $field) {
			$id =  $this->option_name . '_field_mapping_'.$field['shortname'];

			if (trim($field['description']) !=='') {
				$title = $field['description'];
			}else{
				$title = $field['name'];
			}

			if ($field['datatype'] == 'checkbox') {
				$title .= '<span>(only accepts true or false)</span>';
			}

			add_settings_field(
		        $id, 
		        __($title, 'slms-sso'), 
		        array( $this, $this->option_name .'_customfield_cb'),
		        $this->plugin_name,
		        $this->option_name . '_field_mapping',
		        array( 
		        	'label_for' => $id,
		        	'name' => $id,
		        	'id' => $id,
		        )
		    );

		    register_setting( 
		    	$this->plugin_name, 
		    	$id
		    );
		}
	}

	public function get_user_meta_keys(){
		//$keys = array_keys( get_user_meta( get_current_user_id() ) );

		$keys = [
			'nickname','first_name','last_name','description'
		];
		
		$return = '<p>The SSO integration is capable of sending other than email, first and last name. Below is a list of profile fields from your ScholarLMS site that can be mapped with the Wordpress site. You need to insert shortcodes available from your Wordpress site to map a field. The available shortcodes are listed below which can be clicked to copy. For more information, please visit the documentation page.</p>';
		foreach ($keys as $key) {
			$t = "[GETUSERMETA key='".$key."']";
			$return .= '<span class="scbtn" id="sc-'.$key.'"data-clipboard-text ="'.$t.'">'.$t.'</span>';
		}
		$return .= '
		<script src="https://cdnjs.cloudflare.com/ajax/libs/clipboard.js/2.0.8/clipboard.min.js"></script><script src="https://unpkg.com/@popperjs/core@2"></script><script src="https://unpkg.com/tippy.js@6"></script>
		<script>var clipboard = new ClipboardJS(".scbtn");
		clipboard.on("success", function(e) {
		    e.clearSelection();
		});
	    tippy(".scbtn", {
		  content: "Copied",
		  trigger: "click",
		});
		tippy(".scbtn", {
		  content: "Copy",
		  trigger: "mouseenter",
		});
		</script>
		<style>.scbtn{border: 1px solid #a7a5a5;padding: 5px;cursor: pointer;line-height: 3;border-radius: 2px;margin: 5px;white-space: nowrap;}.form-table{margin-bottom:30px;}</style>';
		return $return;
	}
	
    public function slms_uuid_sync_page() {
        ?>
        <div class="wrap">
            <h2>Sync UUIDs from LMS</h2>
            <form id="uuid-method-form">
                <p>Select how users should be matched in Moodle:</p>
                <label><input type="radio" name="match_method" value="email" checked> By Email</label><br>
                <label><input type="radio" name="match_method" value="username"> By Username</label><br><br>
                <button type="submit" class="button button-primary">Start Sync</button>
                <button type="button" class="button button-secondary" id="reset-sync-btn" style="margin-left:10px;">Reset Sync Status</button>
            </form>
    
            <div id="uuid-sync-status" style="display:none;"><p>Initializing...</p></div>
            <div id="uuid-progress-bar" style="width:100%;background:#eee;height:25px;margin-top:10px;display:none;">
                <div style="width:0%;height:100%;background:#0073aa;text-align:center;color:#fff;" id="uuid-progress-inner">0%</div>
            </div>
            <ul id="uuid-sync-log" style="
                margin-top: 15px;
                max-height: 300px;
                overflow-y: auto;
                background: #111;
                color: #eee;
                font-family: monospace;
                font-size: 13px;
                padding: 10px;
                border: 1px solid #444;
                border-radius: 4px;
                display: none;
            "></ul>
            
            <button id="retry-sync" class="button" style="margin-top: 10px; display: none;">Retry Sync</button>
            <script>
            jQuery(function($) {
                let matchMethod = 'email';

                $('#uuid-method-form').on('submit', function(e) {
                    e.preventDefault();
                    matchMethod = $('input[name="match_method"]:checked').val();
                
                    $('#uuid-method-form').hide();
                    $('#uuid-sync-status').show().html('<p><strong>Status:</strong> Initializing...</p>');
                    $('#uuid-progress-bar').show();
                    $('#uuid-sync-log').show().html(''); // clear old logs
                
                    let page = 0;
                    let ajaxurl = "<?php echo admin_url('admin-ajax.php'); ?>";
                
                    function appendLog(msg, isError = false) {
                        const $log = $('#uuid-sync-log');
                        const item = $('<li>').text(msg);
                        if (isError) item.addClass('log-error');
                        $log.append(item);
                        $log.scrollTop($log[0].scrollHeight);
                    }
                
                    function syncPage() {
                        $.post(ajaxurl, {
                            action: "slms_uuid_sync",
                            page: page,
                            method: matchMethod
                        }, function(res) {
                            if (res && res.success) {
                                if (res.data.logs && Array.isArray(res.data.logs)) {
                                    res.data.logs.forEach(log => appendLog(log));
                                }
                
                                $("#uuid-progress-inner").css("width", res.data.percent + "%").text(res.data.percent + "%");
                                $("#uuid-sync-status").html('<p><strong>Status:</strong> Processing...</p>');
                
                                if (res.data.done) {
                                    $("#uuid-sync-status").html(`
                                        <div class='notice notice-success'><p><strong>Status:</strong> UUID sync complete!</p></div>
                                        <a href="<?php echo admin_url('admin.php?page=slms-sso'); ?>" class="button button-secondary" style="margin-top: 10px;">Go Back to Settings</a>
                                    `);
                                } else {
                                    page++;
                                    syncPage();
                                }
                            } else {
                                const error = res && res.data ? res.data : 'Unknown error';
                                $("#uuid-sync-status").html("<div class='notice notice-error'><p><strong>Status:</strong> Sync failed: " + error + "</p></div>");
                                appendLog('Sync failed: ' + error, true);
                            }
                        }).fail(function(jqXHR, textStatus, errorThrown) {
                            const errMsg = `AJAX request failed: ${textStatus} - ${errorThrown}`;
                            $("#uuid-sync-status").html("<div class='notice notice-error'><p><strong>Status:</strong> " + errMsg + "</p></div>");
                            appendLog(errMsg, true);
                            appendLog(jqXHR.responseText || 'No additional error details.', true);
                            $('#retry-sync').show();
                        });
                    }
                
                    syncPage();
                    $('#retry-sync').on('click', function() {
                        location.reload(); // or you could reset vars and call syncPage() again
                    });
                });

                
                $('#reset-sync-btn').on('click', function() {
                    if (!confirm('Are you sure you want to reset sync status?')) return;
                
                    $.post(ajaxurl, {
                        action: 'slms_reset_uuid_sync_status',
                        nonce: '<?php echo wp_create_nonce("slms_uuid_reset_nonce"); ?>'
                    }, function(res) {
                        if (res.success) {
                            alert('UUID sync status has been reset. You can now rerun the sync.');
                            location.reload();
                        } else {
                            alert('Failed to reset: ' + res.data);
                        }
                    });
                });

            });
            
            </script>
        </div>
        <?php
    }

    public function slms_uuid_sync() {
        $page = isset($_POST['page']) ? intval($_POST['page']) : 0;
        $perpage = 20;
        $method = ($_POST['method'] ?? 'email') === 'username' ? 'username' : 'email';
        $logs = [];
    
        $ws = new Slms_Sso_Ws('slms-sso', '3.0.0');
        
        $moodle_response = $ws->call('get_users_without_uid', [
            'page' => $page,
            'perpage' => $perpage
        ]);
        
        if (empty($moodle_response) || !is_array($moodle_response)) {
            $logs[] = 'All users have been synced. No more users returned from Moodle.';
            
            //Set sync complete flag
            update_option('slms_uuid_sync_complete', true);
        
            $this->slms_sso_log($logs);
            wp_send_json_success([
                'done' => true,
                'percent' => 100,
                'logs' => $logs,
                'message' => 'UUID sync complete.'
            ]);
        }
    
        $users = $moodle_response;
        $token = get_option('slmssso_ws_token');
        $url = rtrim(get_option('slmssso_lms_url'), '/');
    
        foreach ($users as $user) {
            $matchvalue = $method === 'username' ? $user['username'] : $user['email'];
            $wp_user = $method === 'username'
                ? get_user_by('login', $matchvalue)
                : get_user_by('email', $matchvalue);
    
            if (!$wp_user) {
                $logs[] = "No matching WP user for $method: $matchvalue";
                continue;
            }
    
            // Get or create UUID
            $uuid = get_user_meta($wp_user->ID, 'slms_sso_uid', true);
            if (empty($uuid)) {
                $uuid = wp_generate_uuid4();
                update_user_meta($wp_user->ID, 'slms_sso_uid', $uuid);
                $logs[] = "Generated UUID for WP user $matchvalue → $uuid";
            } else {
                $logs[] = "WP user $matchvalue already has UUID → $uuid";
            }
    
            // Send UUID to Moodle
            $params = [
                'fieldname'   => 'slms_sso_uid',
                'fieldvalue'  => $uuid,
                'matchfield'  => $method,
                'matchvalue'  => $matchvalue,
            ];
    
            $res = wp_remote_post("$url/webservice/rest/server.php?wstoken=$token&wsfunction=auth_slmswpsso_update_user_uid&moodlewsrestformat=json", [
                'body' => $params
            ]);
    
            if (!is_wp_error($res) && $res['response']['code'] == 200) {
                $logs[] = "Synced {$matchvalue} → UUID: {$uuid}";
            } else {
                $error = is_wp_error($res)
                    ? $res->get_error_message()
                    : (isset($res['body']) ? $res['body'] : 'Invalid response');
                $logs[] = "Failed to sync {$matchvalue}: $error";
            }
        }
    
        // Estimate progress (can be replaced with real Moodle count if available)
        $total_users = 1000;
        $percent = round(min(100, (($page + 1) * $perpage) / $total_users * 100));
    
        $this->slms_sso_log($logs);
    
        wp_send_json_success([
            'done' => false,
            'percent' => $percent,
            'logs' => $logs
        ]);
    }

    
    private function slms_sso_log($logs) {
        $log_file = WP_CONTENT_DIR . '/slms-sso-log.txt';
        $message = is_array($logs) ? implode("\n", $logs) : $logs;
        error_log("[" . date("Y-m-d H:i:s") . "] " . $message . "\n\n", 3, $log_file);
    }
    
    public function reset_uuid_sync_status() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
    
        if (!check_ajax_referer('slms_uuid_reset_nonce', 'nonce', false)) {
            wp_send_json_error('Invalid nonce');
        }
    
        delete_option('slms_uuid_sync_complete');
    
        wp_send_json_success('UUID sync status reset.');
    }

}
