<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       none
 * @since      1.0.0
 *
 * @package    Slms_Sso
 * @subpackage Slms_Sso/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Slms_Sso
 * @subpackage Slms_Sso/public
 * @author     Art Dingal <artdingal@bitkea.com>
 */
class Slms_Sso_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	private $option_name = 'slmssso';

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $lms_plugin_path = '/auth/slmswpsso/wplms.php';

	private $lms_url;
	private $secretkey;
	private $wstoken;
	private $redirecturl;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->lms_url = get_option('slmssso_lms_url');
		$this->secretkey = get_option('slmssso_secret_key');
		$this->wstoken = get_option('slmssso_ws_token');
		$this->redirecturl = get_option('slmssso_redirect_url');
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Slms_Sso_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Slms_Sso_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/slms-sso-public.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Slms_Sso_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Slms_Sso_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/slms-sso-public.js', array( 'jquery' ), $this->version, false );

	}

	/**
	 * Add hidden field in login form
	 *
	 * @since    1.0.0
	 */
	public function add_login_fields() {
	    ?>
	 <input type="hidden" name="slmssso-action" id="slmssso-action" value="login">
	    <?php
	}


	public function add_login_fields_filter() {
		return '<input type="hidden" name="slmssso-action" id="slmssso-action" value="login">';
	}

	/**
	 * Check action and redirect
	 *
	 * @since    1.0.0
	 */
	public function possibly_redirect(){

	    if(isset($_POST['slmssso-action']) && $_POST['slmssso-action'] === 'login') {
	       $this->lms_login(); 
	    }

	    if(isset($_GET['slmssso-action']) && $_GET['slmssso-action'] === 'wplogin'){
	        $this->wp_login();
	    }
	    
	    if(isset($_GET['slmssso-action']) && $_GET['slmssso-action'] === 'wplogout'){
	        $this->wp_logout();
	    }
	}

	/**
	 * login request from LMS
	 * when user login to LMS
	 * @since    1.0.0
	 */
	public function wp_login(){
		$url = $this->lms_url;
	    if ( empty( $url ) ) return;

	    $data = $this->validate_token();
	    $user = get_user_by( 'email', $data['email'] );
	    $redirect_to = isset($_GET['lmswantsurl']) ? $_GET['lmswantsurl'] : '';

	    if ( $user ) {
	        wp_set_current_user( $user->ID, $user->data->user_login );
	        wp_set_auth_cookie( $user->ID );
	        do_action( 'wp_login', $user->data->user_login, $user );
	        
	        if ($redirect_to) {
	            $url = $redirect_to;
	        }
	        
	    }
	    wp_redirect( $url );
	    exit;
	}

	/**
	 * Logout request from LMS
	 * when user logout in LMS
	 * @since    1.0.0
	 */
	public function wp_logout(){
	    $data = $this->validate_token();
	    $url = $this->lms_url;
	    if ( !$url )return;
	    if ($data) {
	        wp_logout();
	    }
	}

	/**
	 * Login user to LMS
	 * when user login to wp
	 * @since    1.0.0
	 */
	public function lms_login(){
	    $response = null;
	    $result = wp_signon();
	    if(is_wp_error($result)) return;
	    
	    $first_name = get_user_meta( $result->data->ID,'first_name');
	    $last_name = get_user_meta( $result->data->ID,'last_name');
	    $user = [
	    	'email' => $result->data->user_email,
	    	'firstname' => $first_name[0],
	    	'lastname' => $last_name[0],
	    	'username' => strtolower($result->data->user_login),
	    	'password' => $_POST['pwd']
	    ];
	    $_SESSION['slmsssoid'] = $result->data->ID;
	    $ws = new Slms_Sso_Ws($this->plugin_name,$this->version);

	    //get fields
	    $fields = $ws->call('request_user_fields');
	    $defaultfields = [];
	    foreach ($fields[0]['defaultfields'] as $key => $field) {
	    	$option =  $this->option_name . '_field_mapping_'.$field['name'];
			$mapped = get_option($option);
			if ($mapped) {
				$val = do_shortcode($mapped);
				$defaultfields[$field['name']] = $val;
			}
		}

		$customfields = [];

		foreach ($fields[0]['customfields'] as $key => $field) {
			$option =  $this->option_name . '_field_mapping_'.$field['shortname'];
		    $mapped = get_option($option);
			if ($mapped) {
				$val = do_shortcode($mapped);
				$customfields[$field['shortname']] = $val;
			}
		}

	    // Only generate UUID if not already set
		$existing_uuid = get_user_meta($result->data->ID, 'slms_sso_uid', true);
		if (empty($existing_uuid)) {
		    if (!function_exists('wp_generate_uuid4')) {
		        require_once ABSPATH . WPINC . '/functions.php';
		    }
		    $uuid = wp_generate_uuid4();
		    update_user_meta($result->data->ID, 'slms_sso_uid', $uuid);
		} else {
		    $uuid = $existing_uuid;
		}
		$user['slms_sso_uid'] = $uuid;
		
		$user['fields'] = json_encode(['defaultfields'=>$defaultfields,'customfields'=>$customfields]);
	    $request_param = array('user' => $user );
		$response = $ws->call('request_login_url',$request_param);
		unset($_SESSION['slmsssoid']);
		$redirecturl = empty(get_option('slmssso_redirect_url')) ? site_url() : get_option('slmssso_redirect_url');
	    if ($response && isset($response['loginurl'])) {
	        wp_redirect( esc_url( $response['loginurl'] ).'&wantsurl='.$redirecturl );exit;
	    }
	}

	/**
	 * Logout user to LMS
	 * when user logout in wp.
	 * @since    1.0.0
	 */
	public function lms_logout(){
	    $key =  $this->secretkey;
	    $url = $this->lms_url;

	    if ( empty( $key ) && empty( $url )) return;

	    $encryption = new Encryption($key);
	    $data = array(
	        'action' => 'lmslogout',
	        'secretkey' => $key,
	    );
	    $token = $encryption->encrypt($data);
	    wp_redirect( $url.$this->lms_plugin_path.'?action=lmslogout&token='.$token );exit;
	}

	/**
	 * validate token from LMS
	 * 
	 * @since    1.0.0
	 */
	public function validate_token(){
	    $token = isset($_GET['token']) ? $_GET['token'] : '';
	    $key = $this->secretkey;
	    if (! empty( $token ) && ! empty( $key )) {    
	        $encryption = new Encryption($key);
	        $data = $encryption->decrypt($token);
	        if ($data['secretkey'] == $key) {
	            return $data;
	        }
	    }
	    return null;
	}

	public function usermeta_shortcode($atts = []){
		$atts = shortcode_atts(
	        array(
	            'key' => '',
	        ), $atts
	    );
	    if ($atts['key'] == '') return '';

		$user_id = $_SESSION['slmsssoid'];
		return get_user_meta($user_id, $atts['key'],true);
	}
}
